/** @file   barrellobject.cpp
 * @brief   Implementation of BarrellObject - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "barrellobject.h"
#include "redrawqueue.h"
#include "gameanims.h"
#include "soundsamples.h"
#include "settings.h"
#include "warglobals.h"
#include "barrellcontroller.h"
#include "bullettable.h"
#include "objectmessage.h"
#include "barrellcontroller.h"
#include "animplayer.h"
#include <fstream>
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
BarrellObject::BarrellObject() :
  GameObject()
{
  this->setCollisionPoint( 0, Vec2D(-30,-15 ) );
  this->setCollisionPoint( 1, Vec2D( 30,-15 ) );
  this->setCollisionPoint( 2, Vec2D( 30, 15 ) );
  this->setCollisionPoint( 3, Vec2D(-30, 15 ) );
  
  int oid = ObjectID::TYPE_BARREL;
  iVelocity.vy = - Settings::floatObjProp( oid, "roll_spd:" );
  this->boundingSphere( Settings::floatObjProp( oid, "bounding_sphere:" ) );
  this->setArmor( 1 );
  iFragmentCount = Settings::intObjProp( oid, "explode_fragments:" );
  iFragmentDamage = Settings::intObjProp( oid, "explode_damage:" );
  
  this->setCorrectAnimation( GameAnims::EIdle );
  
  this->setController( new BarrellController( this ) );
}



/** Destructor
 */
BarrellObject::~BarrellObject()
{
  std::list< ModifyItem* >::iterator iter = iModifyTiles.begin();
  while ( iter != iModifyTiles.end() ) {
    delete (*iter);
    iter++;
  }
  iModifyTiles.clear();
}




///
/// Public methods
/// ==============

/** Updates
 */
void BarrellObject::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  BaseController* pC = this->getController();
  pC->update();
  
  if ( pC->forward() != 0 ) {
    Vec2D mvec( iVelocity );
    mvec.rotate( this->angle() );
    
    this->setCorrectAnimation( GameAnims::EWalk );
    this->move( mvec );
    
    if ( this->collisionNextTime() == true ) {
      // We collided to the tilemap so we explode
      this->kill();
    }
    
  } else {
    this->setCorrectAnimation( GameAnims::EIdle );
  }
}



/** Redraws
 */
void BarrellObject::redraw( RedrawQueue* aQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  
  BITMAP* spriteGfx = m_animation.currentFrame().asBitmap();
  int bx = m_position.intX() - Map::scrollX;
  int by = m_position.intY() - Map::scrollY;
  bx -= spriteGfx->w / 2;
  by -= spriteGfx->h / 2;
  aQueue->addRotatedSprite( RedrawQueue::PRI_ABOVE_NORMAL, bx, by,
                            this->angle(), spriteGfx );
}



/** Kills this object
 */
void BarrellObject::kill()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  // We emit lots of bullets around us
  float angleStep = 256.0 / static_cast<float>( iFragmentCount );
  float a = 0;
  for ( int i=0; i < iFragmentCount; i++ ) {
    Bullet* bullet = BulletTable::createBullet( this, this->position(),
                                                Bullet::EGrenade );
    Vec2D mvec = bullet->velocity();
    mvec.rotate( a );
    bullet->setVelocity( mvec );
    bullet->iDamage = iFragmentDamage;
    
    a += angleStep;
    WarGlobals::pBulletManager->spawnBullet( bullet );
  }
  
  this->setHealth( 0 );
  this->hidden( true );
  this->state( GameObject::STATE_KILLED );
  this->makeSound( GameObject::SND_DIE );
  
  const Animation& explosionAnim = GameAnims::findAnimation( AnimId::KExplosionGrenade);
  AnimPlayer::spawn( explosionAnim, this->position(), 0 );
  
  this->applyMapModification();
}



/** Makes sound
 */
void BarrellObject::makeSound( GameObject::SoundID aSoundId ) const
{
  switch ( aSoundId ) {
    case ( GameObject::SND_DIE ): {
      Sound::playSample( SMP_GRENADE, false );
      break;
    }
    case ( GameObject::SND_PAIN ): {
      Sound::playSample( SMP_METALHIT, false );
      break;
    }
    case ( GameObject::SND_WAKE ): {
      Sound::playSample( SMP_BARRELROLL, false );
      break;
    }
  }
}



/** Handles the bullet hits
 */
bool BarrellObject::hitByBullet( Bullet* aBullet )
{
  if ( aBullet->iOwner != 0 ) {
    if ( aBullet->iOwner->objectType() == ObjectID::TYPE_PLAYER ) {
      // This bullet was shot by a player.
      // We start to roll.
      BarrellController* pC =
        dynamic_cast<BarrellController*>( this->getController() );
      if ( pC->state() != BarrellController::KRolling ) {
        pC->setState( BarrellController::KRolling );
        this->makeSound( GameObject::SND_WAKE );
      }
      
      // Create sparks
      ParticleSparks* pP = new ParticleSparks( aBullet->iPosition,
                                               aBullet->velocity(), 12 );
      WarGlobals::pPartManager->addSystem( pP );
      this->makeSound( GameObject::SND_PAIN );
      return true;
    }
  }

  return false;
}



/** The messageport
 */
void BarrellObject::messagePort( const ObjectMessage& aMessage )
{
  if ( aMessage.m_id == ObjectMessage::KResponse ) {
    return;
  }
  
  // Create a response if possible.
  if ( aMessage.m_pSender != 0 ) {
    ObjectMessage tmpMessage = ObjectMessage::createResponse( aMessage );
    aMessage.m_pSender->messagePort( tmpMessage );
  }
  
  switch ( aMessage.m_id ) {
    case ( ObjectMessage::KControllerState ): {
      int newState = aMessage.m_params.getInt( "message_data:" );
      
      BarrellController* controller =
        dynamic_cast<BarrellController*>( this->getController() );
        
      controller->setState( newState );
      break;
    }
  }
}



/** Reads the modification data from given file.
 */
int BarrellObject::readModifyData( const std::string& aFilename )
{
  if ( aFilename == "NONE" ) {
    return 0;
  }
  
  std::ifstream fin( aFilename.c_str() );
  if ( !fin ) {
    return -1;
  }
  
  if ( Utils::searchForString( fin, "<modify>" ) == true ) {
    fin.close();
    return -1;
  }
  
  while ( true ) {
    if ( fin.eof() == true ) {
      fin.close();
      return -1;
    }
    
    std::string tmp;
    fin >> tmp;
    if ( tmp == "</modify>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      fin.ignore( 4096, '\n' );
      
    } else if ( tmp == "at:" ) {
      // Some temporary variables where we read the data.
      std::string tmpPosX;
      std::string tmpPosY;
      std::string tmpTile;
      
      fin >> tmpPosX >> tmpPosY;
      fin >> tmp;   // read the "set:" - string
      fin >> tmpTile;
      
      ModifyItem* item = new ModifyItem;
      item->iPosX = atoi( tmpPosX.c_str() );
      item->iPosY = atoi( tmpPosY.c_str() );
      item->iNewTile = atoi( tmpTile.c_str() );
      iModifyTiles.push_back( item );
      
    }
  }
  fin.close();
  return 0;
}




///
/// Getter methods
/// ==============

/** Returns the type of us.
 */
ObjectID::Type BarrellObject::objectType() const
{
  return ObjectID::TYPE_BARREL;
}



/** Are we in reloading state
 */
bool BarrellObject::reloading() const
{
  return false;
}




///
/// Private or Protected methods
/// ============================

/** Sets the animation
 */
void BarrellObject::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() == aAnimId ) {
    return;
  }
  const Animation& anim = GameAnims::findAnimation( AnimId::KRollingBarrel, aAnimId );
  this->setAnimation( anim, aAnimId );
}



/** Tells if we're about to collide to the tilemap next time.
 */
bool BarrellObject::collisionNextTime()
{
  Vec2D currentPos( this->position() );
  
  Vec2D mvec( iVelocity );
  mvec.rotate( this->angle() );
  bool ret = this->move( mvec );
  this->position( currentPos );
  return ret;
}



/** Modifies the tiles
 */
void BarrellObject::applyMapModification()
{
  Map::selectLayer( Map::EBackground );
  std::list< ModifyItem* >::iterator iter = iModifyTiles.begin();
  while ( iter != iModifyTiles.end() ) {
    ModifyItem* item = *iter;
    Map::setBlock( item->iPosX, item->iPosY, Map::IN_BLOCKS, item->iNewTile );
    iter++;
  }
}


} // end of namespace
